#!/usr/bin/env node
/**
 * eCripto Server - Headless Node
 *
 * Runs the P2P network node without GUI.
 * For use on Linux servers without display.
 *
 * Usage:
 *   node server.js [--setup] [--port PORT]
 *
 * Options:
 *   --setup    Run first-time setup (create admin account)
 *   --port     Override default P2P port (default: 41235)
 */

const path = require('path');
const fs = require('fs');
const os = require('os');
const readline = require('readline');
const crypto = require('crypto');
const { spawn } = require('child_process');
const https = require('https');
const http = require('http');

// Server version
const SERVER_VERSION = '1.1.0';

// Update check URLs
const UPDATE_URLS = [
  'https://ecripto.app/downloads/check.json',
  'https://raywonderis.me/uploads/website_specific/apps/ecripto/updates/check.json'
];

// Plugin manager (loaded later)
let pluginManager = null;

// Credentials from setup for auto-login
let setupCredentials = null;

// Determine data directory
const getDataDir = () => {
  const platform = process.platform;
  let baseDir;

  if (platform === 'darwin') {
    baseDir = path.join(os.homedir(), 'Library', 'Application Support', 'ecripto');
  } else if (platform === 'win32') {
    baseDir = path.join(process.env.APPDATA || os.homedir(), 'ecripto');
  } else {
    baseDir = path.join(os.homedir(), '.config', 'ecripto');
  }

  const dataDir = path.join(baseDir, 'ecripto-data');

  // Ensure directory exists
  if (!fs.existsSync(dataDir)) {
    fs.mkdirSync(dataDir, { recursive: true });
  }

  return dataDir;
};

const DATA_DIR = getDataDir();

// Load modules
const ECriptoNode = require('./ecripto-node');
const WebUI = require('./web-ui');

// Parse command line arguments
const args = process.argv.slice(2);
const isSetup = args.includes('--setup');
const noAuth = args.includes('--no-auth');
const webOnly = args.includes('--web');
const portIndex = args.indexOf('--port');
const customPort = portIndex !== -1 ? parseInt(args[portIndex + 1]) : null;
const webPortIndex = args.indexOf('--web-port');
const webPortArg = webPortIndex !== -1 ? args[webPortIndex + 1] : null;
const webPort = webPortArg === 'random' || webPortArg === '0' ? 0 : (webPortArg ? parseInt(webPortArg) : 8888);

if (args.includes('--help') || args.includes('-h')) {
  console.log(`
eCripto Server v1.0.0

Usage: node server.js [options]

Options:
  --setup          Run first-time setup (create admin account)
  --no-auth        Start server without requiring CLI login
  --web            Start with web UI only (manage via browser)
  --web-port NUM   Web UI port (default: 8888, use 0 or 'random' for random)
  --port NUM       Override default P2P port (default: 41235)
  --help, -h       Show this help

Examples:
  node server.js --setup          # First-time setup via CLI
  node server.js                  # Normal start (CLI login required)
  node server.js --web            # Start with web UI at http://localhost:8888
  node server.js --web --web-port 9000   # Web UI on custom port
  node server.js --no-auth        # Start without login (CLI only)
`);
  process.exit(0);
}

// Readline interface for CLI (only if stdin is a TTY)
let rl = null;
if (process.stdin.isTTY) {
  rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout
  });
}

const question = (prompt) => new Promise((resolve) => {
  if (rl) {
    rl.question(prompt, resolve);
  } else {
    resolve('');
  }
});

const hideInput = (prompt) => new Promise((resolve) => {
  if (!process.stdin.isTTY) {
    resolve('');
    return;
  }

  process.stdout.write(prompt);
  const stdin = process.stdin;
  const wasRaw = stdin.isRaw;

  stdin.setRawMode(true);
  stdin.resume();
  stdin.setEncoding('utf8');

  let password = '';
  const onData = (char) => {
    if (char === '\n' || char === '\r' || char === '\u0004') {
      stdin.setRawMode(wasRaw);
      stdin.removeListener('data', onData);
      console.log();
      resolve(password);
    } else if (char === '\u0003') {
      process.exit();
    } else if (char === '\u007F' || char === '\b') {
      password = password.slice(0, -1);
      process.stdout.clearLine(0);
      process.stdout.cursorTo(0);
      process.stdout.write(prompt + '*'.repeat(password.length));
    } else {
      password += char;
      process.stdout.write('*');
    }
  };

  stdin.on('data', onData);
});

// Validate password requirements
const validatePassword = (password) => {
  if (password.length < 8) return 'Password must be at least 8 characters';
  if (!/[A-Z]/.test(password)) return 'Password must include uppercase letters';
  if (!/[a-z]/.test(password)) return 'Password must include lowercase letters';
  if (!/[0-9]/.test(password)) return 'Password must include numbers';
  return null;
};

// Setup wizard
const runSetup = async () => {
  console.log('\n========================================');
  console.log('  eCripto Server - First Time Setup');
  console.log('========================================\n');

  // Create admin auth instance
  const AdminAuth = require('./auth/admin-auth');
  const adminAuth = new AdminAuth(DATA_DIR);
  await adminAuth.initialize();

  // Check if already set up
  const authFile = path.join(DATA_DIR, 'admin-auth.json');
  let createNewAdmin = true;

  if (fs.existsSync(authFile) && adminAuth.config?.adminUsers?.length > 0) {
    console.log('Admin account already exists.');
    console.log('');
    console.log('Options:');
    console.log('  1. Keep existing admin and continue to login');
    console.log('  2. Add another admin account');
    console.log('  3. Overwrite existing admin (reset)');
    console.log('  4. Cancel setup');
    console.log('');

    const choice = await question('Choose an option (1-4): ');

    switch (choice.trim()) {
      case '1':
        // Keep existing, skip to service setup
        createNewAdmin = false;
        console.log('\nKeeping existing admin account.');
        break;
      case '2':
        // Add another admin
        createNewAdmin = true;
        break;
      case '3':
        // Overwrite - clear existing admins
        console.log('\nResetting admin accounts...');
        adminAuth.config.adminUsers = [];
        adminAuth.config.initialized = false;
        await adminAuth.saveConfig();
        createNewAdmin = true;
        break;
      case '4':
      default:
        console.log('Setup cancelled.');
        return false;
    }
  }

  if (createNewAdmin) {
    // Get username
    let username = '';
    while (username.length < 3) {
      username = await question('Username (min 3 characters): ');
      if (username.length < 3) {
        console.log('Username must be at least 3 characters.');
      }
    }

    // Check if username already exists
    if (adminAuth.config.adminUsers.find(u => u.username.toLowerCase() === username.toLowerCase())) {
      console.log('Username already exists. Please choose a different username.');
      return false;
    }

    // Get password
    let password = '';
    let passwordError = 'initial';
    while (passwordError) {
      password = await hideInput('Password (min 8 chars, upper/lower/number): ');
      passwordError = validatePassword(password);
      if (passwordError) {
        console.log(passwordError);
      }
    }

    // Confirm password
    const confirmPassword = await hideInput('Confirm password: ');
    if (password !== confirmPassword) {
      console.log('Passwords do not match. Setup cancelled.');
      return false;
    }

    // Get device name
    let deviceName = await question('Device name (e.g., "Main Server"): ');
    if (!deviceName.trim()) {
      deviceName = os.hostname();
    }

    console.log('\nCreating admin account...');

    try {
      // Use createAdmin for additional admins, createInitialAdmin for first admin
      if (adminAuth.config.adminUsers.length === 0) {
        await adminAuth.createInitialAdmin(username, password, deviceName);
      } else {
        await adminAuth.createAdmin(username, password, { deviceName, role: 'admin' });
      }
      console.log('\nAdmin account created successfully!');
      console.log(`Username: ${username}`);
      console.log(`Device: ${deviceName}`);
      console.log(`Data directory: ${DATA_DIR}`);

      // Store credentials for auto-login
      setupCredentials = { username, password };
    } catch (error) {
      console.error('Failed to create admin account:', error.message);
      return false;
    }
  }

  // Ask about systemd service setup (Linux only)
  if (process.platform === 'linux') {
    console.log('\n----------------------------------------');
    console.log('  Systemd Service Setup');
    console.log('----------------------------------------\n');
    console.log('Would you like to install eCripto as a systemd service?');
    console.log('This allows the server to start automatically on boot.\n');
    console.log('Options:');
    console.log('  1. Install as user service (recommended, no sudo needed)');
    console.log('  2. Install as system service (requires sudo)');
    console.log('  3. Skip service installation');
    console.log('');

    const serviceChoice = await question('Choose an option (1-3): ');

    if (serviceChoice.trim() === '1' || serviceChoice.trim() === '2') {
      const asUser = serviceChoice.trim() === '1';
      try {
        console.log(`\nInstalling ${asUser ? 'user' : 'system'} service...`);
        const installPath = process.cwd();
        await installService(installPath, asUser);
        await enableService(asUser);
        console.log('Service installed successfully!');
        console.log(`\nTo manage the service:`);
        if (asUser) {
          console.log('  Start:   systemctl --user start ecripto');
          console.log('  Stop:    systemctl --user stop ecripto');
          console.log('  Status:  systemctl --user status ecripto');
          console.log('  Logs:    journalctl --user -u ecripto -f');
        } else {
          console.log('  Start:   sudo systemctl start ecripto');
          console.log('  Stop:    sudo systemctl stop ecripto');
          console.log('  Status:  sudo systemctl status ecripto');
          console.log('  Logs:    journalctl -u ecripto -f');
        }
      } catch (error) {
        console.error('Failed to install service:', error.message);
        console.log('You can set up the service later via the web UI.');
      }
    } else {
      console.log('\nSkipping service installation.');
      console.log('You can set up the service later via the web UI.');
    }
  }

  return true;
};

// Login prompt
const login = async (adminAuth) => {
  console.log('\n========================================');
  console.log('  eCripto Server - Login');
  console.log('========================================\n');

  // Make sure adminAuth is initialized
  if (!adminAuth.config) {
    await adminAuth.initialize();
  }

  let attempts = 0;
  const maxAttempts = 5;

  while (attempts < maxAttempts) {
    const username = await question('Username: ');
    const password = await hideInput('Password: ');

    try {
      const result = await adminAuth.login(username, password);
      if (result.success) {
        console.log('\nLogin successful!');
        return result.session?.sessionId || 'session';
      } else if (result.requires2FA) {
        const code = await question('Enter 2FA code: ');
        const verify = await adminAuth.verify2FA(result.tempToken, code);
        if (verify.success) {
          console.log('\nLogin successful!');
          return verify.sessionToken;
        }
      } else {
        attempts++;
        console.log(`Login failed: ${result.error || 'Unknown error'}`);
        if (attempts < maxAttempts) {
          console.log(`${maxAttempts - attempts} attempts remaining.\n`);
        }
      }
    } catch (error) {
      attempts++;
      console.log(`Login failed: ${error.message}`);
      if (attempts < maxAttempts) {
        console.log(`${maxAttempts - attempts} attempts remaining.\n`);
      }
    }
  }

  console.log('Too many failed attempts. Exiting.');
  return null;
};

// Check for updates
const checkForUpdates = () => {
  return new Promise((resolve) => {
    let resolved = false;

    const tryUrl = (url) => {
      const protocol = url.startsWith('https') ? https : http;
      const req = protocol.get(url, { timeout: 5000 }, (res) => {
        let data = '';
        res.on('data', chunk => data += chunk);
        res.on('end', () => {
          if (!resolved) {
            resolved = true;
            try {
              const updateInfo = JSON.parse(data);
              resolve({
                currentVersion: SERVER_VERSION,
                latestVersion: updateInfo.version || updateInfo.server?.version,
                updateAvailable: (updateInfo.version || updateInfo.server?.version) !== SERVER_VERSION,
                downloadUrl: updateInfo.server?.downloadUrl || updateInfo.downloadUrl,
                changelog: updateInfo.changelog || updateInfo.server?.changelog,
                checkTime: new Date().toISOString()
              });
            } catch (e) {
              resolve({ error: 'Failed to parse update info', currentVersion: SERVER_VERSION });
            }
          }
        });
      });
      req.on('error', () => {});
      req.on('timeout', () => req.destroy());
    };

    // Try all URLs
    UPDATE_URLS.forEach(tryUrl);

    // Timeout after 10 seconds
    setTimeout(() => {
      if (!resolved) {
        resolved = true;
        resolve({ error: 'Update check timed out', currentVersion: SERVER_VERSION });
      }
    }, 10000);
  });
};

// Download and apply update
const downloadUpdate = async (downloadUrl, installPath) => {
  return new Promise((resolve, reject) => {
    const protocol = downloadUrl.startsWith('https') ? https : http;
    const tempFile = path.join(os.tmpdir(), 'ecripto-update.tar.gz');
    const file = fs.createWriteStream(tempFile);

    protocol.get(downloadUrl, (res) => {
      if (res.statusCode === 302 || res.statusCode === 301) {
        // Follow redirect
        protocol.get(res.headers.location, (res2) => {
          res2.pipe(file);
          file.on('finish', () => {
            file.close();
            resolve(tempFile);
          });
        }).on('error', reject);
      } else {
        res.pipe(file);
        file.on('finish', () => {
          file.close();
          resolve(tempFile);
        });
      }
    }).on('error', reject);
  });
};

// Apply update from tarball
const applyUpdate = async (tarballPath, installPath) => {
  return new Promise((resolve, reject) => {
    // Extract tarball to install path
    const tar = spawn('tar', ['-xzf', tarballPath, '--strip-components=1', '-C', installPath]);

    tar.on('close', (code) => {
      // Clean up temp file
      try { fs.unlinkSync(tarballPath); } catch (e) {}

      if (code === 0) {
        resolve({ success: true });
      } else {
        reject(new Error(`tar extraction failed with code ${code}`));
      }
    });

    tar.on('error', reject);
  });
};

// Restart the server
const restartServer = () => {
  console.log('Restarting server...');

  // Get current script path and args
  const scriptPath = process.argv[1];
  const args = process.argv.slice(2);

  // Spawn new process
  const child = spawn(process.execPath, [scriptPath, ...args], {
    detached: true,
    stdio: 'ignore'
  });

  child.unref();

  // Exit current process
  process.exit(0);
};

// Systemd service management
const SYSTEMD_SERVICE_PATH = '/etc/systemd/system/ecripto.service';
const SYSTEMD_USER_SERVICE_PATH = path.join(os.homedir(), '.config/systemd/user/ecripto.service');

const generateSystemdService = (installPath, isUser = false) => {
  const nodeExec = process.execPath;
  const serverScript = path.join(installPath, 'server.js');
  const workDir = installPath;
  const user = os.userInfo().username;

  return `[Unit]
Description=eCripto P2P Node
After=network.target

[Service]
Type=simple
${isUser ? '' : `User=${user}\n`}WorkingDirectory=${workDir}
ExecStart=${nodeExec} ${serverScript} --web --web-port 8888
Restart=on-failure
RestartSec=10
StandardOutput=syslog
StandardError=syslog
SyslogIdentifier=ecripto
Environment=NODE_ENV=production

[Install]
WantedBy=${isUser ? 'default.target' : 'multi-user.target'}
`;
};

const getServiceStatus = async () => {
  return new Promise((resolve) => {
    const systemService = fs.existsSync(SYSTEMD_SERVICE_PATH);
    const userService = fs.existsSync(SYSTEMD_USER_SERVICE_PATH);

    if (!systemService && !userService) {
      resolve({ installed: false });
      return;
    }

    // Check if enabled
    const checkCmd = spawn('systemctl', [userService ? '--user' : '', 'is-enabled', 'ecripto'].filter(Boolean));
    let output = '';
    checkCmd.stdout.on('data', (d) => output += d.toString());
    checkCmd.on('close', (code) => {
      resolve({
        installed: true,
        isUserService: userService,
        enabled: output.trim() === 'enabled',
        servicePath: userService ? SYSTEMD_USER_SERVICE_PATH : SYSTEMD_SERVICE_PATH
      });
    });
    checkCmd.on('error', () => resolve({ installed: systemService || userService, enabled: false }));
  });
};

const installService = async (asUser = true) => {
  const installPath = path.dirname(process.argv[1]);
  const servicePath = asUser ? SYSTEMD_USER_SERVICE_PATH : SYSTEMD_SERVICE_PATH;
  const serviceContent = generateSystemdService(installPath, asUser);

  // Create directory if needed (for user service)
  if (asUser) {
    const serviceDir = path.dirname(SYSTEMD_USER_SERVICE_PATH);
    if (!fs.existsSync(serviceDir)) {
      fs.mkdirSync(serviceDir, { recursive: true });
    }
  }

  // Write service file
  fs.writeFileSync(servicePath, serviceContent);

  // Reload systemd
  return new Promise((resolve, reject) => {
    const reload = spawn('systemctl', [asUser ? '--user' : '', 'daemon-reload'].filter(Boolean));
    reload.on('close', (code) => {
      if (code === 0) {
        resolve({ success: true, servicePath });
      } else {
        reject(new Error('Failed to reload systemd'));
      }
    });
    reload.on('error', reject);
  });
};

const enableService = async (asUser = true) => {
  return new Promise((resolve, reject) => {
    const enable = spawn('systemctl', [asUser ? '--user' : '', 'enable', 'ecripto'].filter(Boolean));
    enable.on('close', (code) => {
      if (code === 0) {
        resolve({ success: true });
      } else {
        reject(new Error('Failed to enable service'));
      }
    });
    enable.on('error', reject);
  });
};

const disableService = async (asUser = true) => {
  return new Promise((resolve, reject) => {
    const disable = spawn('systemctl', [asUser ? '--user' : '', 'disable', 'ecripto'].filter(Boolean));
    disable.on('close', (code) => {
      if (code === 0) {
        resolve({ success: true });
      } else {
        reject(new Error('Failed to disable service'));
      }
    });
    disable.on('error', reject);
  });
};

// Cached sudo password for session (stored in memory only)
let cachedSudoPassword = null;
let sudoPasswordValid = false;

// Check if user has sudo access (passwordless or cached)
const checkSudoAccess = () => {
  return new Promise((resolve) => {
    // First check if we have a cached valid password
    if (sudoPasswordValid && cachedSudoPassword) {
      resolve(true);
      return;
    }

    // Check for passwordless sudo
    const check = spawn('sudo', ['-n', 'true']);
    check.on('close', (code) => {
      resolve(code === 0);
    });
    check.on('error', () => resolve(false));
  });
};

// Validate sudo password
const validateSudoPassword = (password) => {
  return new Promise((resolve) => {
    const proc = spawn('sudo', ['-S', '-v'], { stdio: ['pipe', 'pipe', 'pipe'] });

    let stderr = '';
    proc.stderr.on('data', (d) => stderr += d.toString());

    proc.on('close', (code) => {
      if (code === 0) {
        cachedSudoPassword = password;
        sudoPasswordValid = true;
        resolve({ success: true });
      } else {
        cachedSudoPassword = null;
        sudoPasswordValid = false;
        resolve({ success: false, error: 'Invalid password' });
      }
    });

    proc.on('error', (err) => {
      resolve({ success: false, error: err.message });
    });

    // Send password to stdin
    proc.stdin.write(password + '\n');
    proc.stdin.end();
  });
};

// Clear cached sudo password
const clearSudoPassword = () => {
  cachedSudoPassword = null;
  sudoPasswordValid = false;
  // Also invalidate the sudo timestamp
  spawn('sudo', ['-k']);
  return { success: true };
};

// Run command with sudo (using cached password if needed)
const runWithSudo = (command, args = []) => {
  return new Promise((resolve, reject) => {
    // Try passwordless first
    const tryPasswordless = spawn('sudo', ['-n', command, ...args]);
    let stdout = '';
    let stderr = '';

    tryPasswordless.stdout.on('data', (d) => stdout += d.toString());
    tryPasswordless.stderr.on('data', (d) => stderr += d.toString());

    tryPasswordless.on('close', (code) => {
      if (code === 0) {
        resolve({ success: true, stdout, stderr });
        return;
      }

      // Passwordless failed, try with cached password
      if (cachedSudoPassword && sudoPasswordValid) {
        const proc = spawn('sudo', ['-S', command, ...args], { stdio: ['pipe', 'pipe', 'pipe'] });
        stdout = '';
        stderr = '';

        proc.stdout.on('data', (d) => stdout += d.toString());
        proc.stderr.on('data', (d) => stderr += d.toString());

        proc.on('close', (code2) => {
          if (code2 === 0) {
            resolve({ success: true, stdout, stderr });
          } else {
            // Password might have expired
            sudoPasswordValid = false;
            reject(new Error(stderr || `Command failed with code ${code2}`));
          }
        });

        proc.on('error', reject);

        // Send password
        proc.stdin.write(cachedSudoPassword + '\n');
        proc.stdin.end();
      } else {
        reject(new Error('Sudo access required. Please enter your password in the web UI.'));
      }
    });

    tryPasswordless.on('error', reject);
  });
};

// Get all network interfaces and IPs
const getNetworkInfo = () => {
  const interfaces = os.networkInterfaces();
  const result = {
    interfaces: [],
    publicIPs: [],
    privateIPs: []
  };

  for (const [name, nets] of Object.entries(interfaces)) {
    for (const net of nets) {
      if (net.family === 'IPv4' && !net.internal) {
        const info = {
          interface: name,
          address: net.address,
          netmask: net.netmask,
          mac: net.mac
        };
        result.interfaces.push(info);

        // Check if private IP
        if (net.address.startsWith('10.') ||
            net.address.startsWith('192.168.') ||
            net.address.match(/^172\.(1[6-9]|2[0-9]|3[0-1])\./)) {
          result.privateIPs.push(net.address);
        } else {
          result.publicIPs.push(net.address);
        }
      }
    }
  }

  return result;
};

// Check for SSH keys and authorized_keys
const getSSHKeyInfo = () => {
  const sshDir = path.join(os.homedir(), '.ssh');
  const result = {
    hasSSHDir: false,
    keys: [],
    hasAuthorizedKeys: false,
    authorizedKeyCount: 0
  };

  if (!fs.existsSync(sshDir)) {
    return result;
  }

  result.hasSSHDir = true;

  // Check for private keys
  const keyFiles = ['id_rsa', 'id_ed25519', 'id_ecdsa', 'id_dsa'];
  for (const keyFile of keyFiles) {
    const keyPath = path.join(sshDir, keyFile);
    const pubPath = path.join(sshDir, keyFile + '.pub');
    if (fs.existsSync(keyPath)) {
      result.keys.push({
        type: keyFile.replace('id_', '').toUpperCase(),
        hasPrivate: true,
        hasPublic: fs.existsSync(pubPath)
      });
    }
  }

  // Check authorized_keys
  const authKeysPath = path.join(sshDir, 'authorized_keys');
  if (fs.existsSync(authKeysPath)) {
    result.hasAuthorizedKeys = true;
    try {
      const content = fs.readFileSync(authKeysPath, 'utf8');
      result.authorizedKeyCount = content.split('\n').filter(line => line.trim() && !line.startsWith('#')).length;
    } catch (e) {}
  }

  return result;
};

// Read nginx configs that may exist (with or without sudo)
const readNginxConfigs = async () => {
  const configs = {
    confD: [],
    sitesAvailable: [],
    sitesEnabled: [],
    canRead: false,
    needsSudo: false
  };

  const tryReadDir = async (dirPath, targetArray) => {
    try {
      // Try direct read first
      if (fs.existsSync(dirPath)) {
        const files = fs.readdirSync(dirPath);
        for (const file of files) {
          if (file.endsWith('.conf') || !file.includes('.')) {
            try {
              const content = fs.readFileSync(path.join(dirPath, file), 'utf8');
              const serverNameMatch = content.match(/server_name\s+([^;]+);/);
              targetArray.push({
                file,
                path: path.join(dirPath, file),
                serverName: serverNameMatch ? serverNameMatch[1].trim() : null
              });
              configs.canRead = true;
            } catch (e) {
              // Can't read this file - need sudo
              targetArray.push({ file, path: path.join(dirPath, file), needsSudo: true });
              configs.needsSudo = true;
            }
          }
        }
      }
    } catch (e) {
      // Directory exists but can't read - need sudo
      configs.needsSudo = true;
    }
  };

  await tryReadDir('/etc/nginx/conf.d', configs.confD);
  await tryReadDir('/etc/nginx/sites-available', configs.sitesAvailable);
  await tryReadDir('/etc/nginx/sites-enabled', configs.sitesEnabled);

  return configs;
};

// Get domains from various sources
const detectDomains = async () => {
  const domains = {
    fromNginx: [],
    fromApache: [],
    fromHosts: [],
    fromHostname: os.hostname()
  };

  // Try to read nginx configs
  try {
    const nginxConfigs = await readNginxConfigs();
    for (const conf of [...nginxConfigs.confD, ...nginxConfigs.sitesAvailable]) {
      if (conf.serverName && !conf.needsSudo) {
        const names = conf.serverName.split(/\s+/);
        domains.fromNginx.push(...names.filter(n => n !== '_' && n !== 'localhost'));
      }
    }
  } catch (e) {}

  // Try to read /etc/hosts
  try {
    const hostsContent = fs.readFileSync('/etc/hosts', 'utf8');
    const lines = hostsContent.split('\n');
    for (const line of lines) {
      if (line.trim() && !line.startsWith('#')) {
        const parts = line.split(/\s+/);
        if (parts.length > 1) {
          for (let i = 1; i < parts.length; i++) {
            if (parts[i] && parts[i] !== 'localhost' && parts[i].includes('.')) {
              domains.fromHosts.push(parts[i]);
            }
          }
        }
      }
    }
  } catch (e) {}

  return domains;
};

// Detect installed services/control panels
const detectInstalledServices = async () => {
  const services = {
    nginx: false,
    apache: false,
    cpanel: false,
    plesk: false,
    webmin: false,
    systemd: false
  };

  // Check for nginx
  try {
    await new Promise((resolve, reject) => {
      const check = spawn('which', ['nginx']);
      check.on('close', (code) => code === 0 ? resolve() : reject());
      check.on('error', reject);
    });
    services.nginx = true;
  } catch (e) {}

  // Check for Apache
  try {
    await new Promise((resolve, reject) => {
      const check = spawn('which', ['apache2', 'httpd']);
      check.on('close', (code) => code === 0 ? resolve() : reject());
      check.on('error', reject);
    });
    services.apache = true;
  } catch (e) {}

  // Check for cPanel
  services.cpanel = fs.existsSync('/usr/local/cpanel');

  // Check for Plesk
  services.plesk = fs.existsSync('/usr/local/psa');

  // Check for Webmin
  services.webmin = fs.existsSync('/etc/webmin');

  // Check for systemd
  services.systemd = fs.existsSync('/run/systemd/system');

  return services;
};

// Nginx configuration management
const NGINX_CONF_PATH = '/etc/nginx/conf.d/ecripto.conf';
const NGINX_SITES_PATH = '/etc/nginx/sites-available/ecripto';

const generateNginxConfig = (domain, port = 8888, ssl = false) => {
  let config = `# eCripto Nginx Configuration
# Generated by eCripto Server

server {
    listen 80;
    server_name ${domain};

    location / {
        proxy_pass http://127.0.0.1:${port};
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;

        proxy_connect_timeout 60s;
        proxy_send_timeout 60s;
        proxy_read_timeout 60s;
    }

    location /ws {
        proxy_pass http://127.0.0.1:${port};
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection "upgrade";
        proxy_set_header Host $host;
        proxy_read_timeout 86400;
    }
}
`;
  return config;
};

const installNginxConfig = async (domain, port = 8888) => {
  const hasSudo = await checkSudoAccess();
  if (!hasSudo) {
    throw new Error('Sudo access required to install nginx config');
  }

  const config = generateNginxConfig(domain, port);
  const tempFile = path.join(os.tmpdir(), 'ecripto-nginx.conf');

  // Write to temp file
  fs.writeFileSync(tempFile, config);

  // Copy to nginx conf.d
  const confPath = fs.existsSync('/etc/nginx/conf.d') ? NGINX_CONF_PATH : NGINX_SITES_PATH;
  await runWithSudo('cp', [tempFile, confPath]);

  // If using sites-available, create symlink
  if (confPath === NGINX_SITES_PATH) {
    try {
      await runWithSudo('ln', ['-sf', NGINX_SITES_PATH, '/etc/nginx/sites-enabled/ecripto']);
    } catch (e) {}
  }

  // Test nginx config
  await runWithSudo('nginx', ['-t']);

  // Reload nginx
  await runWithSudo('systemctl', ['reload', 'nginx']);

  // Clean up
  fs.unlinkSync(tempFile);

  return { success: true, configPath: confPath };
};

const removeNginxConfig = async () => {
  const hasSudo = await checkSudoAccess();
  if (!hasSudo) {
    throw new Error('Sudo access required');
  }

  // Remove config files
  try {
    await runWithSudo('rm', ['-f', NGINX_CONF_PATH]);
    await runWithSudo('rm', ['-f', NGINX_SITES_PATH]);
    await runWithSudo('rm', ['-f', '/etc/nginx/sites-enabled/ecripto']);
  } catch (e) {}

  // Reload nginx
  await runWithSudo('systemctl', ['reload', 'nginx']);

  return { success: true };
};

const getNginxStatus = async () => {
  const confExists = fs.existsSync(NGINX_CONF_PATH) || fs.existsSync(NGINX_SITES_PATH);

  let domain = null;
  let running = false;

  if (confExists) {
    // Try to read domain from config
    try {
      const confPath = fs.existsSync(NGINX_CONF_PATH) ? NGINX_CONF_PATH : NGINX_SITES_PATH;
      const content = fs.readFileSync(confPath, 'utf8');
      const match = content.match(/server_name\s+([^;]+);/);
      if (match) {
        domain = match[1].trim();
      }
    } catch (e) {}
  }

  // Check if nginx is running
  try {
    await new Promise((resolve, reject) => {
      const check = spawn('pgrep', ['nginx']);
      check.on('close', (code) => code === 0 ? resolve() : reject());
      check.on('error', reject);
    });
    running = true;
  } catch (e) {}

  return { installed: confExists, domain, running };
};

// Export for web UI
module.exports = {
  SERVER_VERSION,
  checkForUpdates,
  downloadUpdate,
  applyUpdate,
  restartServer,
  getInstallPath: () => path.dirname(process.argv[1]),
  getServiceStatus,
  installService,
  enableService,
  disableService,
  checkSudoAccess,
  validateSudoPassword,
  clearSudoPassword,
  runWithSudo,
  getNetworkInfo,
  getSSHKeyInfo,
  readNginxConfigs,
  detectDomains,
  detectInstalledServices,
  generateNginxConfig,
  installNginxConfig,
  removeNginxConfig,
  getNginxStatus
};

// Main server function
const startServer = async () => {
  console.log('\n========================================');
  console.log('  eCripto Server v' + SERVER_VERSION);
  console.log('  Decentralized P2P Node');
  console.log('========================================\n');

  console.log(`Data directory: ${DATA_DIR}`);
  console.log(`Platform: ${process.platform}`);
  console.log(`Node.js: ${process.version}`);
  console.log();

  // Check if setup is needed
  const authFile = path.join(DATA_DIR, 'admin-auth.json');
  if (!fs.existsSync(authFile) || isSetup) {
    const setupSuccess = await runSetup();
    if (!setupSuccess) {
      if (rl) rl.close();
      process.exit(1);
    }
  }

  // Initialize admin auth
  const AdminAuth = require('./auth/admin-auth');
  const adminAuth = new AdminAuth(DATA_DIR);
  await adminAuth.initialize();

  // Web UI mode - start web server and skip CLI login
  if (webOnly) {
    console.log('Starting in Web UI mode...');
    console.log(`\nOpen your browser to: http://localhost:${webPort}\n`);

    // Initialize node
    const node = new ECriptoNode(DATA_DIR);
    try {
      await node.initialize();
      await node.start();
      console.log('Node started successfully.\n');
    } catch (error) {
      console.error('Failed to start node:', error.message);
    }

    // Initialize plugins
    console.log('Loading plugins...');
    try {
      const { initializePlugins } = require('./plugins');
      pluginManager = await initializePlugins({
        dataPath: path.join(DATA_DIR, 'plugins'),
        releasePath: process.cwd(),
        nodeId: node.nodeId || crypto.randomBytes(16).toString('hex'),
        nodeName: os.hostname(),
        currentVersion: SERVER_VERSION,
        nodeType: 'server',
        updateServers: UPDATE_URLS.map(u => u.replace('check.json', 'check.php')),
        node: node // Pass node reference for wallet operations
      });
      // Set node reference for wallet routes
      pluginManager.setNode(node);
      console.log('Plugins loaded successfully.\n');
    } catch (error) {
      console.error('Failed to load plugins:', error.message);
    }

    // Start web UI with plugin routes
    const webUI = new WebUI(node, adminAuth, {
      port: webPort,
      pluginRoutes: pluginManager?.getRoutes() || {}
    });
    webUI.start();

    // Handle shutdown
    const shutdown = async () => {
      console.log('\nShutting down...');
      if (pluginManager) pluginManager.stop();
      webUI.stop();
      if (node) await node.stop();
      process.exit(0);
    };

    process.on('SIGINT', shutdown);
    process.on('SIGTERM', shutdown);

    // Keep process alive
    console.log('Press Ctrl+C to stop.\n');
    return;
  }

  // Login (skip if --no-auth or auto-login after setup)
  let sessionToken = null;
  if (noAuth) {
    console.log('Starting without authentication (--no-auth mode)');
    sessionToken = 'no-auth-session';
  } else if (setupCredentials) {
    // Auto-login with credentials from setup
    console.log('\nLogging in automatically...');
    try {
      const result = await adminAuth.login(setupCredentials.username, setupCredentials.password);
      if (result.success) {
        console.log('Login successful!');
        sessionToken = result.session?.sessionId || 'auto-session';
      } else {
        console.log('Auto-login failed, please login manually.');
        sessionToken = await login(adminAuth);
      }
    } catch (error) {
      console.log('Auto-login failed:', error.message);
      sessionToken = await login(adminAuth);
    }
    // Clear credentials from memory
    setupCredentials = null;

    if (!sessionToken) {
      if (rl) rl.close();
      process.exit(1);
    }
  } else {
    sessionToken = await login(adminAuth);
    if (!sessionToken) {
      if (rl) rl.close();
      process.exit(1);
    }
  }

  // Initialize node
  console.log('\nInitializing eCripto node...');

  const node = new ECriptoNode(DATA_DIR);

  // Override port if specified
  if (customPort) {
    console.log(`Using custom port: ${customPort}`);
  }

  try {
    await node.initialize();
    console.log('Node initialized.');

    // Start the node
    await node.start();
    console.log('\neCripto server is running!');
    console.log('Press Ctrl+C to stop.\n');

    // Display status
    const displayStatus = () => {
      const status = node.getStatus();
      console.log('\n--- Node Status ---');
      console.log(`Node ID: ${status.nodeId ? status.nodeId.substring(0, 16) + '...' : 'N/A'}`);
      console.log(`Network: ${status.network?.isRunning ? 'Running' : 'Stopped'}`);
      console.log(`Peers: ${status.network?.peerCount || 0}`);
      console.log(`Blockchain height: ${status.blockchain?.height || 0}`);
      console.log(`Pending transactions: ${status.blockchain?.pendingCount || 0}`);
      console.log('-------------------\n');
    };

    // Show initial status
    displayStatus();

    // Periodic status updates
    const statusInterval = setInterval(displayStatus, 60000); // Every minute

    // Handle commands
    console.log('Commands: status, peers, balance, stop');

    const handleCommand = async (cmd) => {
      const parts = cmd.trim().toLowerCase().split(' ');
      const command = parts[0];

      switch (command) {
        case 'status':
          displayStatus();
          break;

        case 'peers':
          const peers = node.networkManager?.getPeers() || [];
          console.log(`\nConnected peers (${peers.length}):`);
          peers.forEach((p, i) => {
            console.log(`  ${i + 1}. ${p.nodeId?.substring(0, 16)}... @ ${p.address}`);
          });
          console.log();
          break;

        case 'balance':
          const address = node.getWalletAddress();
          const balance = await node.getBalance(address);
          console.log(`\nWallet: ${address}`);
          console.log(`Balance: ${balance} ECR\n`);
          break;

        case 'address':
          console.log(`\nWallet address: ${node.getWalletAddress()}\n`);
          break;

        case 'stop':
        case 'exit':
        case 'quit':
          console.log('\nStopping server...');
          clearInterval(statusInterval);
          await node.stop();
          console.log('Server stopped.');
          if (rl) rl.close();
          process.exit(0);
          break;

        case 'help':
          console.log('\nAvailable commands:');
          console.log('  status  - Show node status');
          console.log('  peers   - List connected peers');
          console.log('  balance - Show wallet balance');
          console.log('  address - Show wallet address');
          console.log('  stop    - Stop the server');
          console.log('  help    - Show this help\n');
          break;

        default:
          if (cmd.trim()) {
            console.log(`Unknown command: ${command}. Type 'help' for available commands.`);
          }
      }
    };

    // Command loop
    if (rl) rl.on('line', handleCommand);

    // Handle shutdown
    const shutdown = async () => {
      console.log('\n\nShutting down...');
      clearInterval(statusInterval);
      await node.stop();
      console.log('eCripto server stopped.');
      if (rl) rl.close();
      process.exit(0);
    };

    process.on('SIGINT', shutdown);
    process.on('SIGTERM', shutdown);

  } catch (error) {
    console.error('Failed to start node:', error.message);
    if (rl) rl.close();
    process.exit(1);
  }
};

// Run
startServer().catch((error) => {
  console.error('Fatal error:', error);
  if (rl) rl.close();
  process.exit(1);
});
